<?php

namespace App\Http\Controllers;

//change the request class if needed
use Illuminate\Http\Request as Request;

use Junges\ACL\Models\Group as Entity;

use Junges\ACL\Models\Permission;

use App\Http\Resources\Select2\PermissionGroup;

use App\Http\Resources\JsonResource;

class GroupsController extends Controller 
{
    public function __construct(Request $request)
    {
        $this->request = $request;
    }
    
    public function index()
    {   
        return view('groups.index');
    }
    
    public function datatable()
    {
        return datatables(Entity::query())
            ->addColumn('actions', function ($entity) {
                return view('groups.partials.table.actions', compact('entity'));
            })
            ->rawColumns(['actions'])
            ->setRowAttr([
                'data-id' => function ($entity) {
                    return $entity->id;
                },
            ])
            ->make(true);
    }
    
    public function create()
    {
        return view('groups.create', [
            'entity' => new Entity(),
            'usedPermissions' => [],
            'permissions' => PermissionGroup::collection(
                Permission::get()->groupBy(function ($item, $key) {
                    return explode(':', $item->quard_name)[0];
                })->values()
            )->toArray($this->request)[0],
        ]);
    }
    
    public function store()
    {
        $request = $this->request;
        
        $data = $request->validate([
            'name' => 'required|string|min:3|max:100',
            'permissions' => 'required|array',
            'permissions.*.*' => 'integer|exists:permissions,id',
            'description' => 'nullable|string|min:10|max:655',
        ]);
        
        $data['quard_name'] = \Str::snake($data['name']);

        $entity = new Entity();
        $entity->fill($data);
        
        $entity->save();
        //popunjavamo niz sa permisijama
        $permissionsArray = [];
        foreach(collect($data['permissions'])->collapse()->toArray() as $perm){
            $permissionsArray[] = Permission::find($perm);
        }
        //syncPermissions brise sve postojece permisije za grupu i dodeljuje grupi nove (koje prosledjujemo u nizu)
        $entity->syncPermissions($permissionsArray);
       
        
        
        if ($request->wantsJson()) {
            return JsonResource::make()->withSuccess(__('Grupa korisnika je uspešno sačuvana!'));
        }
        
        return redirect()->route('groups.index')->withSystemSuccess(__('Grupa korisnika je uspešno sačuvana!'));
    }
    
    
    public function edit()
    {   
        $request = $this->request;
        
        $entity = Entity::find($request->entity);
        
        return view('groups.edit', [
            'entity' => $entity,
            'usedPermissions' => $entity->permissions->pluck('id')->toArray(),
            'permissions' => PermissionGroup::collection(
                Permission::get()->groupBy(function ($item, $key) {
                    return explode(':', $item->quard_name)[0];

                })->values()
            )->toArray($this->request)[0],
        ]);
    }
    
    public function update()
    {   
        $request = $this->request;
        
        $entity = Entity::find($request->entity);
        
        $data = $request->validate([
            'name' => 'required|string|min:3|max:100',
            'permissions' => 'required|array',
            'permissions.*.*' => 'integer|exists:permissions,id',
            'description' => 'nullable|string|min:10|max:655',
        ]);

        $data['quard_name'] = \Str::snake($data['name']);

        
        $groupsPermission = Permission::where('name', 'groups')
                                        ->orWhere('name', 'backoffice users')
                                        ->get()
                                        ->pluck('id')
                                        ->toArray();
        
        if(auth()->user()->hasGroup($entity->quard_name)){

            if(!array_key_exists('backoffice_users', $data['permissions'])){
                $data['permissions']['backoffice_users'] = [];
            }
            $data['permissions']['backoffice_users'] = array_merge($data['permissions']['backoffice_users'], $groupsPermission);
        }
        
        $entity->fill($data);

        $entity->save();
        
        //popunjavamo niz sa permisijama
        $permissionsArray = [];
        foreach(collect($data['permissions'])->collapse()->toArray() as $perm){
            $permissionsArray[] = Permission::find($perm);
        }
        //syncPermissions brise sve postojece permisije za grupu i dodeljuje grupi nove (koje prosledjujemo u nizu)
        $entity->syncPermissions($permissionsArray);

        
        if ($request->wantsJson()) {
            return JsonResource::make()->withSuccess(__('Grupa korisnika je uspešno sačuvana!'));
        }
        
        return redirect()->route('groups.index')->withSystemSuccess(__('Grupa korisnika je uspešno sačuvana!'));
    }
    
    public function delete(Entity $entity)
    {
        foreach($entity->permissions as $permission){
            $entity->revokePermission($permission);
        }

        $entity->delete();
        
        if ($this->request->wantsJson()) {
            return JsonResource::make()->withSuccess(__('Grupa korisnika je uspešno obrisana!'));
        }
        
        return redirect()->route('groups.index')->withSystemSuccess(__('Grupa korisnika je uspešno obrisana!'));
    }
}
