<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request as Request;
use Illuminate\Support\Carbon;
use App\Http\Resources\JsonResource;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use App\Http\Requests\Podesavanja\ProfilRequest;
use App\Rules\jeIspravnaLozinka;


class ProfileController extends Controller
{
    
    /**
     * Prikazi 'view' sa info podacima o profilu 'auth' korisnika i sa
     * formom za izmenu tih podataka.
     */
    public function edit()
    {
        return view('profile.edit');
    }

    /**
     * Izvrsi izmenu profilnih podataka 'auth' korisnika
     */
    public function update(ProfilRequest $request)
    {
        $entity = auth()->user();

        $notificationMessage = __("Profilni podaci su izmenjeni");
        
        $data = $request->validated();
        unset($data['image_profile']);
        
        $entity->update($data);
        
        $entity->updateImage('image_profile');

        if ($request->wantsJson()) {
            return JsonResource::make($entity)->withSuccess($notificationMessage);
        }
        
        return redirect()->route('profile.edit')
                         ->withSystemMessage($notificationMessage);
    }

    /**
     * Prikazi 'view' za izmenu lozinke 
     */
    public function editPassword()
    {
        return view('profile.edit_password');
    }

    /**
     * Izmeni lozinku korisnika
     */
    public function updatePassword(Request $request)
    {
        $entity = auth()->user();

        $data = $request->validate([
            'old_password' => ['required', 'string', 'max:190', new jeIspravnaLozinka()],
            'password' => ['required', 'string', 'min:8', 'max:190', 'confirmed'],
            'password_confirmation' => ['required', 'string'],
        ]);        

        $entity->password = bcrypt($data['password']);
        $entity->save();
        
        return redirect()->route('profile.edit_password')
                         ->withSystemMessage('Lozinka je izmenjena');
    }

    public function editImage()
    {
       // Da li treba ovaj metod za AJAX izmenu slike ???? 
    }

    /*
     * Obrisi sliku preko ajaks poziva
     */ 
    public function deleteImage(Request $request) {

        $request->validate([
            "imageClass" => ["required", "string", "exists:images,class"]
        ]);
        
        $stat = auth()->user()->deleteImage($request->imageClass);
        if(!$stat) {
            return JsonResource::make()->withError(__("Greška. Slika nije mogla biti izbrisana."));
        }
        
        return JsonResource::make()->withSuccess(__("Slika je izbrisana."));
    }
}