<?php

namespace App\Lib;

/**
 * Miscellaneous helper functions
 * Should contain only methods which are independent of a single project.
 *
 */
class HelperPack 
{
    
    /**
     * Throw an error if variable $var isn't of type string
     * 
     * @param string $var      | variable to test
     * @param string $varName  | name of the variable for exception string
     * 
     * @throws \InvalidArgumentException
     * 
     * @return void
     */
    private static function throwErrorIfNotString($var, $varName) {
        if(!is_string($var)) {
            throw new \InvalidArgumentException("String is required as \$$varName parameter");
        }
    }
    
    
    /**
     * Validator method for generateEmailStr() method arguments.
     * 
     * @params same as in generateEmailStr()
     * @throws \InvalidArgumentException
     * 
     * @return void
     */
    private static function validateGenerateEmailStrMethodArgs($domain, $sep, $userData)
    {
        self::throwErrorIfNotString($domain, 'domain');
        self::throwErrorIfNotString($sep, 'domain');
        foreach($userData as $key => $val) {
            self::throwErrorIfNotString($val, '$userData variable arguments ' . $key);
        }
    }
    
    
    /**
     * Generate email string from domain, separator and user data.
     * 
     * @param string $domain  | email domain (e.g. example.com)
     * @param string $sep     | user data separator on 'implode' 
     * @param Array $userData | user data in varible number of arguments 
     * 
     * @throws \InvalidArgumentException
     * 
     * @return string
     */
    public static function generateEmailStr($domain, $sep, ...$userData)
    {
        // allow array to be used instead of variable func arguments
        if(count($userData) == 1 && is_array($userData[0])) {
            $userData = $userData[0];
        }
        
        // validate input arguments
        self::validateGenerateEmailStrMethodArgs($domain, $sep, $userData);
        
        $preparedUserData = array_map("strtolower", $userData);
        $preDomain = implode($sep, $preparedUserData);
        
        return sprintf("%s@%s", $preDomain, $domain);
    }
    
    /**
     * Remove non-alphabetic characters from string. 
     * If mode == 'sql' > create SQL replacement regex string
     * If mode == 'php' > create PHP replacement regex string
     * 
     * @param string $val | Value which will be sanitized by this method
     * @param string $for | 'sql' or 'php'
     * 
     * @return string
     */
    public static function replaceNonAlpha($val, $for='sql')
    {
        if($for == 'sql') {
            return "REGEXP_REPLACE($val,'[-.,(){}\[:blank:\]+*/&%^$#:;|%_~=]+','')";
        }
        else if($for == 'php') {
            return preg_replace('/[-.,(){}\s+*\/&%^$#:;\|%_~=]+/','', $val);
        }
        else {
            throw new \InvalidArgumentException(__METHOD__ 
                    . ': $for parameter value is invalid.');
        }
    }
    
    public static function generateEntitiesPermissionsForSection(
        $entity,
        $entityName,
        $subEntities
    ) {
        $permissions = [
            [
                'name' => $entityName,
                'slug' => $entity
            ],
        ];
        foreach ($subEntities as $subEntity) {
            $permissions[$subEntity] = [
                'name' => $entityName . ' - ' . ucwords(str_replace(str_split('._'), ' ', $subEntity)), 
                'slug' => $entity . ':' . $subEntity,
            ];
        }

        return $permissions;
    }

    /**
     * Clear input data for SQL
     * 
     * @param string $input
     * 
     * @return string
     */
    public static function sanitizeInputSql($input)
    {
        return addslashes($input);//mysqli_real_escape_string($input);
    }
}
